<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\PetugasModel;
use CodeIgniter\Exceptions\PageNotFoundException;
use Myth\Auth\Password;

class DataPetugas extends BaseController
{
    protected PetugasModel $petugasModel;

    protected $petugasValidationRules = [
        'email' => [
            'rules' => 'required|valid_email|is_unique[users.email]',
            'errors' => [
                'required' => 'Email harus diisi.',
                'is_unique' => 'Email ini telah terdaftar.',
                'valid_email' => 'Format email tidak valid.'
            ]
        ],
        'username' => [
            'rules' => 'required|min_length[6]|is_unique[users.username]',
            'errors' => [
                'required' => 'Username harus diisi',
                'is_unique' => 'Username ini telah terdaftar.'
            ]
        ],
        'password' => [
            'rules' => 'permit_empty|min_length[6]',
            'errors' => [
                'min_length' => 'Password minimal 6 karakter'
            ]
        ],
        'role' => [
            'rules' => 'required',
            'errors' => [
                'required' => 'Role wajib diisi'
            ]
        ]
    ];

    public function __construct()
    {
        $this->petugasModel = new PetugasModel();
    }

    public function index()
    {
        if (user()->toArray()['is_superadmin'] != '1') {
            return redirect()->to('admin');
        }

        $data = [
            'title' => 'Data Petugas',
            'ctx' => 'petugas'
        ];

        return view('admin/petugas/data-petugas', $data);
    }

    public function ambilDataPetugas()
    {
        $petugas = $this->petugasModel->getAllPetugas();

        $data = [
            'data' => $petugas,
            'empty' => empty($petugas)
        ];

        return view('admin/petugas/list-data-petugas', $data);
    }

    // Form tampil register petugas
    public function registerPetugas()
    {
        if (user()->toArray()['is_superadmin'] != '1') {
            return redirect()->to('admin');
        }

        $data = [
            'title' => 'Register Petugas',
            'ctx' => 'petugas'
        ];

        return view('admin/petugas/register', $data);
    }

    // Simpan data petugas baru dari form
    public function savePetugas()
    {
        if (user()->toArray()['is_superadmin'] != '1') {
            return redirect()->to('admin');
        }

        if (!$this->validate($this->petugasValidationRules)) {
            return view('admin/petugas/register', [
                'title' => 'Register Petugas',
                'ctx' => 'petugas',
                'validation' => $this->validator,
                'oldInput' => $this->request->getVar()
            ]);
        }

        $email = $this->request->getVar('email');
        $username = $this->request->getVar('username');
        $password = $this->request->getVar('password');
        $passwordHash = $password ? Password::hash($password) : null;
        $role = $this->request->getVar('role'); // Ambil role dari form

        $result = $this->petugasModel->savePetugas(null, $email, $username, $passwordHash, $role);

        if ($result) {
            session()->setFlashdata([
                'msg' => 'Petugas berhasil ditambahkan',
                'error' => false
            ]);
            return redirect()->to('/admin/petugas');
        }

        session()->setFlashdata([
            'msg' => 'Gagal menambahkan petugas',
            'error' => true
        ]);
        return redirect()->back()->withInput();
    }

    public function formEditPetugas($id)
    {
        $petugas = $this->petugasModel->getPetugasById($id);

        if (empty($petugas)) {
            throw new PageNotFoundException('Data petugas dengan id ' . $id . ' tidak ditemukan');
        }

        $data = [
            'data' => $petugas,
            'ctx' => 'petugas',
            'title' => 'Edit Data Petugas',
        ];

        return view('admin/petugas/edit-data-petugas', $data);
    }

    public function updatePetugas()
{
    $idPetugas = $this->request->getVar('id');
    $petugasLama = $this->petugasModel->getPetugasById($idPetugas);

    // RULES KHUSUS UPDATE
    $emailRule = 'required|valid_email';
    if ($petugasLama['email'] != $this->request->getVar('email')) {
        $emailRule .= '|is_unique[users.email,id,' . $idPetugas . ']';
    }

    $usernameRule = 'required|min_length[6]';
    if ($petugasLama['username'] != $this->request->getVar('username')) {
        $usernameRule .= '|is_unique[users.username,id,' . $idPetugas . ']';
    }

    // Update rules sementara
    $this->petugasValidationRules['email']['rules'] = $emailRule;
    $this->petugasValidationRules['username']['rules'] = $usernameRule;

    if (!$this->validate($this->petugasValidationRules)) {
        $data = [
            'data' => $petugasLama,
            'ctx' => 'petugas',
            'title' => 'Edit Data Petugas',
            'validation' => $this->validator,
            'oldInput' => $this->request->getVar()
        ];
        return view('admin/petugas/edit-data-petugas', $data);
    }

    $password = $this->request->getVar('password') ?? false;
    $email = $this->request->getVar('email');
    $username = $this->request->getVar('username');
    $passwordHash = $password ? Password::hash($password) : $petugasLama['password_hash'];
    $role = $this->request->getVar('role');

    $result = $this->petugasModel->savePetugas($idPetugas, $email, $username, $passwordHash, $role);

    if ($result) {
        session()->setFlashdata([
            'msg' => 'Edit data berhasil',
            'error' => false
        ]);
        return redirect()->to('/admin/petugas');
    }

    session()->setFlashdata([
        'msg' => 'Gagal mengubah data',
        'error' => true
    ]);
    return redirect()->to('/admin/petugas/edit/' . $idPetugas);
}


    public function delete($id)
    {
        $result = $this->petugasModel->delete($id);

        if ($result) {
            session()->setFlashdata([
                'msg' => 'Data berhasil dihapus',
                'error' => false
            ]);
            return redirect()->to('/admin/petugas');
        }

        session()->setFlashdata([
            'msg' => 'Gagal menghapus data',
            'error' => true
        ]);
        return redirect()->to('/admin/petugas');
    }
}
