<?php

namespace App\Controllers\Admin\Lap;

use App\Controllers\BaseController;
use App\Models\Lap\BPresensiSiswaModel;
use App\Models\Lap\BKelasModel;
use App\Models\Lap\BSiswaModel;
use App\Models\HariLiburModel;

class LapControllerBulanan extends BaseController
{
    protected $presensiModel;
    protected $kelasModel;
    protected $siswaModel;
    protected $hariLiburModel;

    public function __construct()
    {
        $this->presensiModel   = new BPresensiSiswaModel();
        $this->kelasModel      = new BKelasModel();
        $this->siswaModel      = new BSiswaModel();
        $this->hariLiburModel  = new HariLiburModel();
    }

    public function index()
    {
        $bulan        = $this->request->getGet('bulan') ?? date('Y-m');
        $tanggalAwal  = date('Y-m-01', strtotime($bulan));
        $tanggalAkhir = date('Y-m-t', strtotime($bulan));
        $idKelas      = $this->request->getGet('kelas') ?? 'all';
        $status       = $this->request->getGet('status') ?? 'all';

        $kelasList = $this->kelasModel->getKelas();
        $laporan   = [];

        $siswaList = ($idKelas === 'all')
            ? $this->siswaModel->getSiswaWithKelas()
            : $this->siswaModel->getSiswaByKelas($idKelas);

        $presensiList = $this->presensiModel
            ->getPresensiByTanggalRange($tanggalAwal, $tanggalAkhir, $idKelas);

        $presensiMap = [];
        foreach ($presensiList as $p) {
            $presensiMap[$p['id_siswa']][$p['tanggal']] = $p;
        }

        // Ambil daftar hari libur
        $liburList = $this->hariLiburModel
            ->where("tanggal BETWEEN '$tanggalAwal' AND '$tanggalAkhir'")
            ->findAll();

        $liburMap = [];
        foreach ($liburList as $l) {
            $liburMap[$l['tanggal']] = $l['keterangan'];
        }

        foreach ($siswaList as $s) {

            $hadir = $sakit = $izin = $alpa = $terlambat = $tidakSolat = 0;

            $period = new \DatePeriod(
                new \DateTime($tanggalAwal),
                new \DateInterval('P1D'),
                (new \DateTime($tanggalAkhir))->modify('+1 day')
            );

            foreach ($period as $dt) {
                $tgl = $dt->format('Y-m-d');

                // Skip hari libur
                if (isset($liburMap[$tgl])) continue;

                $p = $presensiMap[$s['id_siswa']][$tgl] ?? null;

                if ($p) {
                    $kehadiran = ($p['id_kehadiran'] ?? '') === '' ? '4' : $p['id_kehadiran'];
                    switch ($kehadiran) {
                        case '1':
                            $hadir++;
                            if (!empty($p['jam_masuk']) && $p['jam_masuk'] > '07:00:00') $terlambat++;
                            if (empty($p['jam_sholat'])) $tidakSolat++;
                            break;
                        case '2': $sakit++; break;
                        case '3': $izin++; break;
                        default: $alpa++;
                    }
                    // Jam masuk kosong → alpa
                    if (empty($p['jam_masuk'])) $alpa++;
                } else {
                    // Tidak ada presensi → alpa
                    $alpa++;
                }
            }

            $row = [
                'id_siswa'    => $s['id_siswa'],
                'nis'         => $s['nis'] ?? '-',
                'nama_siswa'  => $s['nama_siswa'],
                'jk'          => $s['jenis_kelamin'] ?? '-',
                'kelas'       => $s['nama_kelas'] ?? '-',
                'hadir'       => $hadir,
                'sakit'       => $sakit,
                'izin'        => $izin,
                'alpa'        => $alpa,
                'terlambat'   => $terlambat,
                'tidak_solat' => $tidakSolat,
            ];

            if ($status !== 'all') {
                if ($status === 'terlambat' && $terlambat == 0) continue;
                if ($status === 'tidak_solat' && $tidakSolat == 0) continue;
                if (in_array($status, ['1','2','3','4']) &&
                    $row[$this->statusField($status)] == 0) continue;
            }

            $laporan[] = $row;
        }

        $summary = [
            'hadir'       => 0,
            'sakit'       => 0,
            'izin'        => 0,
            'alpa'        => 0,
            'terlambat'   => 0,
            'tidak_solat' => 0,
            'total'       => count($siswaList),
        ];

        foreach ($laporan as $r) {
            $summary['hadir']       += $r['hadir'];
            $summary['sakit']       += $r['sakit'];
            $summary['izin']        += $r['izin'];
            $summary['alpa']        += $r['alpa'];
            $summary['terlambat']   += $r['terlambat'];
            $summary['tidak_solat'] += $r['tidak_solat'];
        }

        return view('admin/lap/lapbulanan', [
            'title'         => 'LAPORAN PRESENSI BULANAN',
            'laporan'       => $laporan,
            'kelasList'     => $kelasList,
            'selectedKelas' => $idKelas,
            'tanggalAwal'   => $tanggalAwal,
            'tanggalAkhir'  => $tanggalAkhir,
            'bulan'         => $bulan,
            'summary'       => $summary,
        ]);
    }

    private function statusField($status)
    {
        return match ($status) {
            '1' => 'hadir',
            '2' => 'sakit',
            '3' => 'izin',
            '4' => 'alpa',
            default => 'hadir'
        };
    }

    public function exportCSV()
    {
        $bulan        = $this->request->getGet('bulan') ?? date('Y-m');
        $tanggalAwal  = date('Y-m-01', strtotime($bulan));
        $tanggalAkhir = date('Y-m-t', strtotime($bulan));
        $idKelas      = $this->request->getGet('kelas') ?? 'all';
        $status       = $this->request->getGet('status') ?? 'all';

        $siswaList = ($idKelas === 'all')
            ? $this->siswaModel->getSiswaWithKelas()
            : $this->siswaModel->getSiswaByKelas($idKelas);

        $presensiList = $this->presensiModel
            ->getPresensiByTanggalRange($tanggalAwal, $tanggalAkhir, $idKelas);

        $presensiMap = [];
        foreach ($presensiList as $p) {
            $presensiMap[$p['id_siswa']][$p['tanggal']] = $p;
        }

        $liburList = $this->hariLiburModel
            ->where("tanggal BETWEEN '$tanggalAwal' AND '$tanggalAkhir'")
            ->findAll();

        $liburMap = [];
        foreach ($liburList as $l) {
            $liburMap[$l['tanggal']] = $l['keterangan'];
        }

        $laporan = [];
        foreach ($siswaList as $s) {

            $hadir = $sakit = $izin = $alpa = $terlambat = $tidakSolat = 0;

            $period = new \DatePeriod(
                new \DateTime($tanggalAwal),
                new \DateInterval('P1D'),
                (new \DateTime($tanggalAkhir))->modify('+1 day')
            );

            foreach ($period as $dt) {
                $tgl = $dt->format('Y-m-d');

                if (isset($liburMap[$tgl])) continue;

                $p = $presensiMap[$s['id_siswa']][$tgl] ?? null;

                if ($p) {
                    $kehadiran = ($p['id_kehadiran'] ?? '') === '' ? '4' : $p['id_kehadiran'];
                    switch ($kehadiran) {
                        case '1':
                            $hadir++;
                            if (!empty($p['jam_masuk']) && $p['jam_masuk'] > '07:00:00') $terlambat++;
                            if (empty($p['jam_sholat'])) $tidakSolat++;
                            break;
                        case '2': $sakit++; break;
                        case '3': $izin++; break;
                        default: $alpa++;
                    }
                    if (empty($p['jam_masuk'])) $alpa++;
                } else {
                    $alpa++;
                }
            }

            $row = [
                'Nama'          => $s['nama_siswa'],
                'NIS'           => $s['nis'] ?? '-',
                'Kelas'         => $s['nama_kelas'] ?? '-',
                'Jenis Kelamin' => $s['jenis_kelamin'] ?? '-',
                'Hadir'         => $hadir,
                'Sakit'         => $sakit,
                'Izin'          => $izin,
                'Alpa'          => $alpa,
                'Terlambat'     => $terlambat,
                'Tidak Solat'   => $tidakSolat,
            ];

            if ($status !== 'all') {
                if ($status === 'terlambat' && $terlambat == 0) continue;
                if ($status === 'tidak_solat' && $tidakSolat == 0) continue;
                if (in_array($status, ['1','2','3','4']) &&
                    $row[$this->statusField($status)] == 0) continue;
            }

            $laporan[] = $row;
        }

        $filename = 'laporan_presensi_'.$bulan.'.csv';
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="'.$filename.'"');

        echo "\xEF\xBB\xBF";
        $output = fopen('php://output', 'w');

        if (!empty($laporan)) {
            fputcsv($output, array_merge(['No'], array_keys($laporan[0])), ';');
            $no = 1;
            foreach ($laporan as $row) {
                fputcsv($output, array_merge([$no++], $row), ';');
            }
        }

        fclose($output);
        exit;
    }
}
