<?php

namespace App\Controllers\Admin\Lap;

use App\Controllers\BaseController;
use App\Models\Lap\HPresensiSiswaModel;
use App\Models\Lap\HKelasModel;
use App\Models\Lap\HSiswaModel;

class LapControllerHarian extends BaseController
{
    protected $presensiModel;
    protected $kelasModel;
    protected $siswaModel;

    public function __construct()
    {
        $this->presensiModel = new HPresensiSiswaModel();
        $this->kelasModel    = new HKelasModel();
        $this->siswaModel    = new HSiswaModel();
    }

    /* ======================================================
     * LAPORAN HARIAN
     * ====================================================== */
    public function index()
    {
        $tanggal = $this->request->getGet('tanggal') ?? date('Y-m-d');
        $idKelas = $this->request->getGet('kelas') ?? 'all';
        $status  = $this->request->getGet('status') ?? 'all';

        $kelasList = $this->kelasModel->getKelas();
        $laporan   = [];

        if ($idKelas) {

            // 1️⃣ DATA SISWA & PRESENSI
            if ($idKelas === 'all') {
                $siswaList    = $this->siswaModel->getSiswaWithKelas();
                $presensiList = $this->presensiModel->getPresensiByTanggal($tanggal);
            } else {
                $siswaList    = $this->siswaModel->getSiswaByKelas($idKelas);
                $presensiList = $this->presensiModel
                    ->getPresensiByKelasTanggal($idKelas, $tanggal);
            }

            // 2️⃣ MAP PRESENSI
            $presensiMap = [];
            foreach ($presensiList as $p) {
                $presensiMap[$p['id_siswa']] = $p;
            }

            // 3️⃣ GABUNG DATA (DEFAULT = ALPA)
            foreach ($siswaList as $s) {

                $p = $presensiMap[$s['id_siswa']] ?? [];

                $row = [
                    'id_siswa'     => $s['id_siswa'],
                    'nis'          => $s['nis'] ?? '-',
                    'nama_siswa'   => $s['nama_siswa'],
                    'jk'           => $s['jenis_kelamin'] ?? '-',
                    'kelas'        => $s['nama_kelas'] ?? '-',
                    'tanggal'      => $tanggal,
                    'jam_masuk'    => $p['jam_masuk'] ?? null,
                    'jam_sholat'   => $p['jam_sholat'] ?? null,
                    'jam_keluar'   => $p['jam_keluar'] ?? null,
                    'id_kehadiran' => $p['id_kehadiran'] ?? '4',
                    'keterangan'   => $p['keterangan'] ?? '',
                ];

// 4️⃣ FILTER STATUS CARD
if ($status !== 'all') {

    // ⏰ TERLAMBAT (jam masuk > 07:00)
    if ($status === 'terlambat') {
        if (empty($row['jam_masuk']) || $row['jam_masuk'] <= '07:00:00') continue;
    }
    // ❌ ALPA
    elseif ($status === '4') {
        if (in_array($row['id_kehadiran'], ['1','2','3'])) continue;
    }
    // STATUS HADIR / SAKIT / IZIN
    else {
        if ($row['id_kehadiran'] !== $status) continue;
    }
}

                $laporan[] = $row;
            }
        }

        return view('admin/lap/lapharian', [
            'title'         => 'LAPORAN PRESENSI SISWA',
            'laporan'       => $laporan,
            'kelasList'     => $kelasList,
            'selectedKelas' => $idKelas,
            'tanggal'       => $tanggal,
        ]);
    }
/* ======================================================
 * EXPORT CSV
 * ====================================================== */
public function exportCSV()
{
    $tanggal = $this->request->getGet('tanggal') ?? date('Y-m-d');
    $idKelas = $this->request->getGet('kelas');

    if (!$idKelas) {
        return redirect()->back()
            ->with('error', 'Pilih kelas terlebih dahulu.');
    }

    if ($idKelas === 'all') {
        $siswaList    = $this->siswaModel->getSiswaWithKelas();
        $presensiList = $this->presensiModel->getPresensiByTanggal($tanggal);
        $filename     = "Laporan_Harian_Semua_Kelas_{$tanggal}.csv";
    } else {
        $siswaList    = $this->siswaModel->getSiswaByKelas($idKelas);
        $presensiList = $this->presensiModel
            ->getPresensiByKelasTanggal($idKelas, $tanggal);
        $filename     = "Laporan_Harian_{$tanggal}.csv";
    }

    $presensiMap = [];
    foreach ($presensiList as $p) {
        $presensiMap[$p['id_siswa']] = $p;
    }

    header('Content-Type: text/csv; charset=UTF-8');
    header("Content-Disposition: attachment; filename=\"$filename\"");
    echo "\xEF\xBB\xBF";

    $output = fopen('php://output', 'w');
    fputcsv($output, [
        'No','Nama','JK','Kelas','Tanggal',
        'Jam Masuk','Keterangan','Jam Sholat','Jam Keluar','Status'
    ], ';');

    $no = 1;
    foreach ($siswaList as $s) {

        $p = $presensiMap[$s['id_siswa']] ?? [];

        $status = match ($p['id_kehadiran'] ?? '4') {
            '1' => 'Hadir',
            '2' => 'Sakit',
            '3' => 'Izin',
            default => 'Alpa',
        };

        // logika Jam Masuk
        $jamMasuk = $p['jam_masuk'] ?? '';

        // logika Keterangan
        $keterangan = '';
        if ($status === 'Hadir' && !empty($jamMasuk)) {
            // bandingkan dengan jam 07:00
            if (strtotime($jamMasuk) > strtotime('07:00')) {
                $keterangan = 'Terlambat';
            } else {
                $keterangan = 'Tepat Waktu';
            }
        }

        // logika Jam Sholat
        $jamSholat = $p['jam_sholat'] ?? '';
        if ($status === 'Hadir' && empty($jamSholat)) {
            $jamSholat = 'Tidak Sholat';
        }

        // logika Jam Keluar
        $jamKeluar = $p['jam_keluar'] ?? '';
        if ($status === 'Hadir' && empty($jamKeluar)) {
            $jamKeluar = 'TAP';
        }

        fputcsv($output, [
            $no++,
            $s['nama_siswa'],
            $s['jenis_kelamin'] ?? '-',
            $s['nama_kelas'] ?? '-',
            $tanggal,
            $jamMasuk,
            $keterangan,
            $jamSholat,
            $jamKeluar,
            $status,
        ], ';');
    }

    fclose($output);
    exit;
}

/* ======================================================
 * DETAIL LAPORAN SISWA (PER BULAN / PER TANGGAL)
 * ====================================================== */
public function detailSiswa($idSiswa)
{
    // =============================
    // PARAM DARI HALAMAN SEBELUMNYA
    // =============================
    $kelas   = $this->request->getGet('kelas');
    $tanggal = $this->request->getGet('tanggal');

    // =============================
    // FILTER DETAIL
    // =============================
    $tanggalAwal  = $this->request->getGet('awal') ?? date('Y-m-01');
    $tanggalAkhir = $this->request->getGet('akhir') ?? date('Y-m-d');

    // =============================
    // DATA SISWA
    // =============================
    $siswa = $this->siswaModel
        ->select('tb_siswa.*, tb_kelas.kelas')
        ->join('tb_kelas', 'tb_kelas.id_kelas = tb_siswa.id_kelas', 'left')
        ->where('id_siswa', $idSiswa)
        ->first();

    if (!$siswa) {
        return redirect()->back()->with('error', 'Siswa tidak ditemukan');
    }

    // =============================
    // DATA PRESENSI
    // =============================
    $presensi = $this->presensiModel
        ->where('id_siswa', $idSiswa)
        ->where('tanggal >=', $tanggalAwal)
        ->where('tanggal <=', $tanggalAkhir)
        ->orderBy('tanggal', 'ASC')
        ->findAll();

    // =============================
    // URL KEMBALI (FINAL)
    // =============================
    $backUrl = site_url(
        'admin/lap/lapharian'
        .'?kelas='.$kelas
        .'&tanggal='.$tanggal
    );

    // =============================
    // VIEW
    // =============================
    return view('admin/lap/lapdetailsiswa', [
        'title'    => 'DETAIL LAPORAN SISWA',
        'siswa'    => $siswa,
        'presensi' => $presensi,
        'awal'     => $tanggalAwal,
        'akhir'    => $tanggalAkhir,
        'back_url' => $backUrl,
    ]);
}









//Can't find a route for 'GET: admin/lap/exportSiswaCSV'.
public function exportSiswaCSVDetail()
{
    $idSiswa = $this->request->getGet('id_siswa');
    $awal    = $this->request->getGet('awal');
    $akhir   = $this->request->getGet('akhir');

    if (!$idSiswa || !$awal || !$akhir) {
        return redirect()->back()->with('error', 'Parameter tidak lengkap');
    }

    // ambil data siswa
    $siswa = $this->siswaModel->find($idSiswa);

    // ambil presensi
    $presensi = $this->presensiModel->getPresensiSiswaRange(
        $idSiswa,
        $awal,
        $akhir
    );

    $filename = "Presensi_{$siswa['nama_siswa']}_{$awal}_{$akhir}.csv";

    header('Content-Type: text/csv; charset=UTF-8');
    header("Content-Disposition: attachment; filename=\"$filename\"");
    echo "\xEF\xBB\xBF";

    $output = fopen('php://output', 'w');

    fputcsv($output, [
        'No',
        'Tanggal',
        'Jam Masuk',
        'Keterangan',
        'Jam Sholat',
        'Jam Pulang',
        'Status'
    ], ';');

    $no = 1;
    foreach ($presensi as $p) {

        $status = match ($p['id_kehadiran']) {
            '1' => 'Hadir',
            '2' => 'Sakit',
            '3' => 'Izin',
            default => 'Alpa'
        };

        // Jam masuk
        $jamMasuk = $p['jam_masuk'] ?? '';

        // Keterangan
        $keterangan = '';
        if ($status === 'Hadir' && $jamMasuk) {
            $keterangan = (strtotime($jamMasuk) > strtotime('07:00'))
                ? 'Terlambat'
                : 'Tepat Waktu';
        }

        // Jam sholat
        $jamSholat = $p['jam_sholat'] ?? '';
        if ($status === 'Hadir' && empty($jamSholat)) {
            $jamSholat = 'Tidak Sholat';
        }

        // Jam pulang
        $jamKeluar = $p['jam_keluar'] ?? '';
        if ($status === 'Hadir' && empty($jamKeluar)) {
            $jamKeluar = 'TAP';
        }

        fputcsv($output, [
            $no++,
            $p['tanggal'],
            $jamMasuk,
            $keterangan,
            $jamSholat,
            $jamKeluar,
            $status
        ], ';');
    }

    fclose($output);
    exit;
}









}
