<?php

namespace App\Controllers\Admin\Persentase;

use App\Controllers\BaseController;
use App\Models\PresensiSiswaModel;
use App\Models\KelasModel;
use App\Models\SiswaModel;

class Persentase extends BaseController
{
    protected $presensiModel;
    protected $kelasModel;
    protected $siswaModel;

    public function __construct()
    {
        $this->presensiModel = new PresensiSiswaModel();
        $this->kelasModel    = new KelasModel();
        $this->siswaModel    = new SiswaModel();
    }

    public function index()
    {
        // ================= INPUT =================
        $tanggal = $this->request->getGet('tanggal') ?? date('Y-m-d');
        $idKelas = $this->request->getGet('kelas');

        // ================= DATA KELAS =================
        $kelas = $this->kelasModel->getAllKelas();

        // default
        $stat      = null;
        $chartData = null;

        if (!empty($idKelas)) {

            // ================= TOTAL SISWA =================
            $totalSiswa = $this->siswaModel
                ->where('id_kelas', $idKelas)
                ->countAllResults();

            // ================= HARIAN =================
            $hadirHarian = $this->presensiModel
                ->where([
                    'id_kelas'      => $idKelas,
                    'tanggal'       => $tanggal,
                    'id_kehadiran'  => 1
                ])
                ->countAllResults();

            // ================= MINGGUAN =================
            $startWeek = date('Y-m-d', strtotime('monday this week', strtotime($tanggal)));
            $endWeek   = date('Y-m-d', strtotime('sunday this week', strtotime($tanggal)));

            $hadirMingguan = $this->presensiModel
                ->where('id_kelas', $idKelas)
                ->where('tanggal >=', $startWeek)
                ->where('tanggal <=', $endWeek)
                ->where('id_kehadiran', 1)
                ->countAllResults();

            // ================= BULANAN =================
            $bulan = date('m', strtotime($tanggal));
            $tahun = date('Y', strtotime($tanggal));

            $hadirBulanan = $this->presensiModel
                ->where('id_kelas', $idKelas)
                ->where('MONTH(tanggal)', $bulan, false)
                ->where('YEAR(tanggal)', $tahun, false)
                ->where('id_kehadiran', 1)
                ->countAllResults();

            // ================= PERSENTASE =================
            // Catatan:
            // - Mingguan diasumsikan 5 hari sekolah
            // - Bulanan diasumsikan 20 hari sekolah

            $persenHarian   = $totalSiswa > 0
                ? round(($hadirHarian / $totalSiswa) * 100, 2)
                : 0;

            $persenMingguan = $totalSiswa > 0
                ? round(($hadirMingguan / ($totalSiswa * 5)) * 100, 2)
                : 0;

            $persenBulanan  = $totalSiswa > 0
                ? round(($hadirBulanan / ($totalSiswa * 20)) * 100, 2)
                : 0;

            // ================= DATA KE VIEW =================
            $stat = [
                'totalSiswa' => $totalSiswa,
                'harian'     => $persenHarian,
                'mingguan'   => $persenMingguan,
                'bulanan'    => $persenBulanan,
            ];

            $chartData = [
                'labels' => ['Harian', 'Mingguan', 'Bulanan'],
                'values' => [$persenHarian, $persenMingguan, $persenBulanan],
            ];
        }

        // ================= RETURN VIEW =================
        return view('admin/persentase/index', [
            'title'         => 'Persentase Kehadiran Siswa',
            'kelas'         => $kelas,
            'selectedKelas' => $idKelas,
            'tanggal'       => $tanggal,
            'stat'          => $stat,
            'chartData'     => $chartData
        ]);
    }


    // ===================== TREND HARIAN =====================
    public function trenHarian()
    {
        $bulan  = $this->request->getGet('bulan') ?? date('m');
        $tahun  = $this->request->getGet('tahun') ?? date('Y');
        $idKelas = $this->request->getGet('kelas');

        $kelas = $this->kelasModel->getAllKelas();
        $stat  = null;
        $chartData = null;

        if (!empty($idKelas)) {

            // total siswa kelas
            $totalSiswa = $this->siswaModel
                ->where('id_kelas', $idKelas)
                ->countAllResults();

            // ambil jumlah hari di bulan
            $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $bulan, $tahun);

            $labels = [];
            $values = [];

            for ($day = 1; $day <= $daysInMonth; $day++) {
                $tanggal = sprintf('%04d-%02d-%02d', $tahun, $bulan, $day);

                // cek hadir hari itu
                $hadirHari = $this->presensiModel
                    ->where('id_kelas', $idKelas)
                    ->where('tanggal', $tanggal)
                    ->where('id_kehadiran', 1)
                    ->countAllResults();

                $persen = $totalSiswa > 0
                    ? round(($hadirHari / $totalSiswa) * 100, 2)
                    : 0;

                $labels[] = $day;
                $values[] = $persen;
            }

            $chartData = [
                'labels' => $labels,
                'values' => $values
            ];
        }

        return view('admin/persentase/tren_harian', [
            'title'         => 'Tren Kehadiran Harian',
            'kelas'         => $kelas,
            'selectedKelas' => $idKelas,
            'bulan'         => $bulan,
            'tahun'         => $tahun,
            'chartData'     => $chartData
        ]);
    }





    // ===================== Perbandingan Antar Kelas =====================
public function perbandinganKelas()
{
    $tanggal = $this->request->getGet('tanggal') ?? date('Y-m-d');

    $kelasList = $this->kelasModel->getAllKelas();
    $chartLabels = [];
    $chartValues = [];
    $persenKelas = [];

    foreach ($kelasList as $kelas) {
        $totalSiswa = $this->siswaModel
            ->where('id_kelas', $kelas['id_kelas'])
            ->countAllResults();

        $hadir = $this->presensiModel
            ->where('id_kelas', $kelas['id_kelas'])
            ->where('tanggal', $tanggal)
            ->where('id_kehadiran', 1)
            ->countAllResults();

        $persen = $totalSiswa > 0 ? round(($hadir / $totalSiswa) * 100, 2) : 0;

        $chartLabels[] = $kelas['kelas'];
        $chartValues[] = $persen;
        $persenKelas[] = [
            'kelas' => $kelas['kelas'],
            'persen' => $persen
        ];
    }

    // urut dari terendah ke tertinggi
    usort($persenKelas, function($a, $b){
        return $a['persen'] <=> $b['persen'];
    });

    return view('admin/persentase/perbandingan_kelas', [
        'title' => 'Perbandingan Antar Kelas',
        'tanggal' => $tanggal,
        'chartLabels' => $chartLabels,
        'chartValues' => $chartValues,
        'persenKelas' => $persenKelas
    ]);
}


// ===================== Perbandingan Kehadiran Sekolah Hari ke Hari =====================
public function perbandinganSekolah()
{
    $startDate = $this->request->getGet('start') ?? date('Y-m-01'); // default awal bulan
    $endDate   = $this->request->getGet('end') ?? date('Y-m-d');     // default hari ini

    $dates = [];
    $values = [];

    $period = new \DatePeriod(
        new \DateTime($startDate),
        new \DateInterval('P1D'),
        (new \DateTime($endDate))->modify('+1 day')
    );

    // total siswa sekolah
    $totalSiswa = $this->siswaModel->countAllResults();

    foreach ($period as $date) {
        $tanggal = $date->format('Y-m-d');
        $hadir = $this->presensiModel
            ->where('tanggal', $tanggal)
            ->where('id_kehadiran', 1)
            ->countAllResults();

        $persen = $totalSiswa > 0 ? round(($hadir / $totalSiswa) * 100, 2) : 0;

        $dates[] = $tanggal;
        $values[] = $persen;
    }

    return view('admin/persentase/perbandingan_sekolah', [
        'title' => 'Perbandingan Kehadiran Sekolah',
        'startDate' => $startDate,
        'endDate'   => $endDate,
        'chartLabels' => $dates,
        'chartValues' => $values
    ]);
}

// ===================== Perbandingan Antar Siswa per Kelas =====================
public function perbandinganSiswa()
{
    $idKelas = $this->request->getGet('kelas');
    $startDate = $this->request->getGet('start') ?? date('Y-m-01');
    $endDate = $this->request->getGet('end') ?? date('Y-m-d');

    $kelasList = $this->kelasModel->getAllKelas();
    $siswaData = [];
    $chartLabels = [];
    $chartValues = [];

    if (!empty($idKelas)) {
        $siswaList = $this->siswaModel
            ->where('id_kelas', $idKelas)
            ->findAll();

        foreach ($siswaList as $siswa) {

            $period = new \DatePeriod(
                new \DateTime($startDate),
                new \DateInterval('P1D'),
                (new \DateTime($endDate))->modify('+1 day')
            );

            $totalHari = 0;
            $hadir = 0;

            foreach ($period as $date) {
                $tanggal = $date->format('Y-m-d');
                $totalHari++;

                $presensi = $this->presensiModel
                    ->where('id_siswa', $siswa['id_siswa'])
                    ->where('tanggal', $tanggal)
                    ->first();

                if ($presensi && $presensi['id_kehadiran'] == 1) {
                    $hadir++;
                }
            }

            $persen = $totalHari > 0 ? round(($hadir / $totalHari) * 100, 2) : 0;

            $chartLabels[] = $siswa['nama_siswa'];
            $chartValues[] = $persen;

            $siswaData[] = [
                'nama' => $siswa['nama_siswa'],
                'persen' => $persen
            ];
        }

        // urut dari kehadiran terendah ke tertinggi
        usort($siswaData, function($a,$b){
            return $a['persen'] <=> $b['persen'];
        });
    }

    return view('admin/persentase/perbandingan_siswa', [
        'title' => 'Perbandingan Antar Siswa per Kelas',
        'kelasList' => $kelasList,
        'selectedKelas' => $idKelas,
        'startDate' => $startDate,
        'endDate' => $endDate,
        'chartLabels' => $chartLabels,
        'chartValues' => $chartValues,
        'siswaData' => $siswaData
    ]);
}




// ===================== Perbandingan Antar Siswa Seluruh Sekolah =====================
public function perbandinganSiswaSekolah()
{
    $startDate = $this->request->getGet('start') ?? date('Y-m-01');
    $endDate   = $this->request->getGet('end') ?? date('Y-m-d');

    $siswaList = $this->siswaModel->findAll();
    $chartLabels = [];
    $chartValues = [];
    $siswaData = [];

    foreach ($siswaList as $siswa) {
        $period = new \DatePeriod(
            new \DateTime($startDate),
            new \DateInterval('P1D'),
            (new \DateTime($endDate))->modify('+1 day')
        );

        $totalHari = 0;
        $hadir = 0;

        foreach ($period as $date) {
            $tanggal = $date->format('Y-m-d');
            $totalHari++;

            $presensi = $this->presensiModel
                ->where('id_siswa', $siswa['id_siswa'])
                ->where('tanggal', $tanggal)
                ->first();

            if ($presensi && $presensi['id_kehadiran'] == 1) {
                $hadir++;
            }
        }

        $persen = $totalHari > 0 ? round(($hadir / $totalHari) * 100, 2) : 0;

        $chartLabels[] = $siswa['nama_siswa'];
        $chartValues[] = $persen;

        $siswaData[] = [
            'nama' => $siswa['nama_siswa'],
            'persen' => $persen
        ];
    }

    // urut dari kehadiran terendah ke tertinggi
    usort($siswaData, function($a,$b){
        return $a['persen'] <=> $b['persen'];
    });

    return view('admin/persentase/perbandingan_siswa_sekolah', [
        'title' => 'Perbandingan Antar Siswa Seluruh Sekolah',
        'startDate' => $startDate,
        'endDate' => $endDate,
        'chartLabels' => $chartLabels,
        'chartValues' => $chartValues,
        'siswaData' => $siswaData
    ]);
}


// ===================== Kelas Terbaik Berdasarkan Kehadiran =====================
public function kelasTerbaik()
{
    $startDate = $this->request->getGet('start') ?? date('Y-m-01');
    $endDate   = $this->request->getGet('end') ?? date('Y-m-d');

    $kelasList = $this->kelasModel->getAllKelas();
    $chartLabels = [];
    $chartValues = [];
    $kelasData = [];

    foreach ($kelasList as $kelas) {
        $siswaList = $this->siswaModel
            ->where('id_kelas', $kelas['id_kelas'])
            ->findAll();

        $totalSiswa = count($siswaList);
        $totalHari = 0;
        $hadir = 0;

        if ($totalSiswa == 0) continue;

        $period = new \DatePeriod(
            new \DateTime($startDate),
            new \DateInterval('P1D'),
            (new \DateTime($endDate))->modify('+1 day')
        );

        foreach ($period as $date) {
            $tanggal = $date->format('Y-m-d');
            $totalHari++;

            foreach ($siswaList as $siswa) {
                $presensi = $this->presensiModel
                    ->where('id_siswa', $siswa['id_siswa'])
                    ->where('tanggal', $tanggal)
                    ->first();

                if ($presensi && $presensi['id_kehadiran'] == 1) {
                    $hadir++;
                }
            }
        }

        $totalPossible = $totalSiswa * $totalHari;
        $persen = $totalPossible > 0 ? round(($hadir / $totalPossible) * 100, 2) : 0;

        $chartLabels[] = $kelas['kelas'];
        $chartValues[] = $persen;

        $kelasData[] = [
            'kelas' => $kelas['kelas'],
            'persen' => $persen
        ];
    }

    // urut dari tertinggi → terendah
    usort($kelasData, function($a,$b){
        return $b['persen'] <=> $a['persen'];
    });

    return view('admin/persentase/kelas_terbaik', [
        'title' => 'Kelas Terbaik Berdasarkan Kehadiran',
        'startDate' => $startDate,
        'endDate' => $endDate,
        'chartLabels' => $chartLabels,
        'chartValues' => $chartValues,
        'kelasData' => $kelasData
    ]);
}




// ===================== Kelas Terbaik Solat Duhur =====================
public function kelasTerbaikSolat()
{
    $startDate = $this->request->getGet('start') ?? date('Y-m-01');
    $endDate   = $this->request->getGet('end') ?? date('Y-m-d');

    $kelasList = $this->kelasModel->getAllKelas();
    $chartLabels = [];
    $chartValues = [];
    $kelasData = [];

    foreach ($kelasList as $kelas) {
        $siswaList = $this->siswaModel
            ->where('id_kelas', $kelas['id_kelas'])
            ->findAll();

        $totalSiswa = count($siswaList);
        $totalHari = 0;
        $hadir = 0;

        if ($totalSiswa == 0) continue;

        $period = new \DatePeriod(
            new \DateTime($startDate),
            new \DateInterval('P1D'),
            (new \DateTime($endDate))->modify('+1 day')
        );

        foreach ($period as $date) {
            $tanggal = $date->format('Y-m-d');
            $totalHari++;

            foreach ($siswaList as $siswa) {
                $presensi = $this->presensiModel
                    ->where('id_siswa', $siswa['id_siswa'])
                    ->where('tanggal', $tanggal)
                    ->where("jam_sholat IS NOT NULL", null, false)
                    ->first();

                if ($presensi) {
                    $hadir++;
                }
            }
        }

        $totalPossible = $totalSiswa * $totalHari;
        $persen = $totalPossible > 0 ? round(($hadir / $totalPossible) * 100, 2) : 0;

        $chartLabels[] = $kelas['kelas'];
        $chartValues[] = $persen;

        $kelasData[] = [
            'kelas' => $kelas['kelas'],
            'persen' => $persen
        ];
    }

    // urut dari tertinggi → terendah
    usort($kelasData, function($a,$b){
        return $b['persen'] <=> $a['persen'];
    });

    return view('admin/persentase/kelas_terbaik_solat', [
        'title' => 'Kelas Terbaik Solat Duhur',
        'startDate' => $startDate,
        'endDate' => $endDate,
        'chartLabels' => $chartLabels,
        'chartValues' => $chartValues,
        'kelasData' => $kelasData
    ]);
}



// ===================== Kelas Paling Banyak Kesiangan =====================
public function kelasPalingTelat()
{
    $startDate = $this->request->getGet('start') ?? date('Y-m-01');
    $endDate   = $this->request->getGet('end') ?? date('Y-m-d');

    $kelasList = $this->kelasModel->getAllKelas();
    $chartLabels = [];
    $chartValues = [];
    $kelasData = [];

    foreach ($kelasList as $kelas) {
        $siswaList = $this->siswaModel
            ->where('id_kelas', $kelas['id_kelas'])
            ->findAll();

        $totalSiswa = count($siswaList);
        $totalHari = 0;
        $telat = 0;

        if ($totalSiswa == 0) continue;

        $period = new \DatePeriod(
            new \DateTime($startDate),
            new \DateInterval('P1D'),
            (new \DateTime($endDate))->modify('+1 day')
        );

        foreach ($period as $date) {
            $tanggal = $date->format('Y-m-d');
            $totalHari++;

            foreach ($siswaList as $siswa) {
                $presensi = $this->presensiModel
                    ->where('id_siswa', $siswa['id_siswa'])
                    ->where('tanggal', $tanggal)
                    ->first();

                if ($presensi && $presensi['jam_masuk'] > '07:00:00') {
                    // asumsikan jam masuk normal 07:00
                    $telat++;
                }
            }
        }

        $totalPossible = $totalSiswa * $totalHari;
        $persenTelat = $totalPossible > 0 ? round(($telat / $totalPossible) * 100, 2) : 0;

        $chartLabels[] = $kelas['kelas'];
        $chartValues[] = $persenTelat;

        $kelasData[] = [
            'kelas' => $kelas['kelas'],
            'persen' => $persenTelat
        ];
    }

    // urut dari paling telat → sedikit telat
    usort($kelasData, function($a,$b){
        return $b['persen'] <=> $a['persen'];
    });

    return view('admin/persentase/kelas_paling_telat', [
        'title' => 'Kelas Paling Banyak Kesiangan',
        'startDate' => $startDate,
        'endDate' => $endDate,
        'chartLabels' => $chartLabels,
        'chartValues' => $chartValues,
        'kelasData' => $kelasData
    ]);
}



// ===================== Kelas Paling Banyak Alpa =====================
public function kelasPalingAlpa()
{
    $startDate = $this->request->getGet('start') ?? date('Y-m-01');
    $endDate   = $this->request->getGet('end') ?? date('Y-m-d');

    $kelasList = $this->kelasModel->getAllKelas();
    $chartLabels = [];
    $chartValues = [];
    $kelasData = [];

    foreach ($kelasList as $kelas) {
        $siswaList = $this->siswaModel
            ->where('id_kelas', $kelas['id_kelas'])
            ->findAll();

        $totalSiswa = count($siswaList);
        $totalHari = 0;
        $alpa = 0;

        if ($totalSiswa == 0) continue;

        $period = new \DatePeriod(
            new \DateTime($startDate),
            new \DateInterval('P1D'),
            (new \DateTime($endDate))->modify('+1 day')
        );

        foreach ($period as $date) {
            $tanggal = $date->format('Y-m-d');
            $totalHari++;

            foreach ($siswaList as $siswa) {
                $presensi = $this->presensiModel
                    ->where('id_siswa', $siswa['id_siswa'])
                    ->where('tanggal', $tanggal)
                    ->first();

                if (!$presensi || $presensi['id_kehadiran'] == 4) {
                    // id_kehadiran 4 = alpa
                    $alpa++;
                }
            }
        }

        $totalPossible = $totalSiswa * $totalHari;
        $persenAlpa = $totalPossible > 0 ? round(($alpa / $totalPossible) * 100, 2) : 0;

        $chartLabels[] = $kelas['kelas'];
        $chartValues[] = $persenAlpa;

        $kelasData[] = [
            'kelas' => $kelas['kelas'],
            'persen' => $persenAlpa
        ];
    }

    // urut dari paling banyak alpa → sedikit
    usort($kelasData, function($a,$b){
        return $b['persen'] <=> $a['persen'];
    });

    return view('admin/persentase/kelas_paling_alpa', [
        'title' => 'Kelas Paling Banyak Alpa',
        'startDate' => $startDate,
        'endDate' => $endDate,
        'chartLabels' => $chartLabels,
        'chartValues' => $chartValues,
        'kelasData' => $kelasData
    ]);
}



}
