<?php

namespace App\Models;

use CodeIgniter\Model;

class PresensiModel extends Model
{
    protected $table = 'tb_presensi_siswa';
    protected $primaryKey = 'id_presensi';



    /**
     * Hitung presensi berdasarkan status kehadiran
     */
    public function countByKehadiran($status, $tanggal)
    {
        return $this->db->table('tb_presensi_siswa p')
            ->join('tb_kehadiran k', 'k.id_kehadiran = p.id_kehadiran')
            ->where('k.kehadiran', $status)
            ->where('p.tanggal', $tanggal)
            ->countAllResults();
    }



    /**
     * Hitung siswa yang ALPHA (belum presensi hari ini)
     */
    public function countAlpha($tanggal)
    {
        return $this->db->table('tb_siswa s')
            ->whereNotIn('s.id_siswa', function ($builder) use ($tanggal) {
                return $builder->select('id_siswa')
                    ->from('tb_presensi_siswa')
                    ->where('tanggal', $tanggal);
            })
            ->countAllResults();
    }


    // // 🔥 ====================================================
    // ==========================================================
    public function getPresensiHarian($tanggal, $id_kelas = null)
    {
        $builder = $this->db->table('tb_siswa s')
            ->select('
                s.nis,
                s.nama_siswa,
                s.jenis_kelamin,
                k.kelas,
                p.jam_masuk,
                p.jam_keluar,
                p.jam_sholat,
                kh.kehadiran,
                p.keterangan
            ')
            ->join('tb_kelas k', 'k.id_kelas = s.id_kelas')
            ->join(
                'tb_presensi_siswa p',
                'p.id_siswa = s.id_siswa AND p.tanggal = "'.$tanggal.'"',
                'left'
            )
            ->join('tb_kehadiran kh', 'kh.id_kehadiran = p.id_kehadiran', 'left');

        if (!empty($id_kelas)) {
            $builder->where('s.id_kelas', $id_kelas);
        }

        return $builder->orderBy('s.nama_siswa', 'ASC')->get()->getResultArray();
    }


// Hadir ============================================================
public function getPresensiHadir($tanggal, $id_kelas = null)
{
    $builder = $this->db->table('tb_siswa s')
        ->select('
            s.nis,
            s.nama_siswa,
            s.jenis_kelamin,
            k.kelas,
            p.jam_masuk,
            p.jam_keluar,
            p.jam_sholat,
            kh.kehadiran,
            p.keterangan
        ')
        ->join('tb_kelas k', 'k.id_kelas = s.id_kelas')
        ->join(
            'tb_presensi_siswa p',
            'p.id_siswa = s.id_siswa AND p.tanggal = "'.$tanggal.'"',
            'left'
        )
        ->join('tb_kehadiran kh', 'kh.id_kehadiran = p.id_kehadiran', 'left')
        ->where('kh.kehadiran','hadir'); // <-- hanya yang hadir (atau izin/sakit)

    if (!empty($id_kelas)) {
        $builder->where('s.id_kelas', $id_kelas);
    }

    return $builder->orderBy('s.nama_siswa', 'ASC')->get()->getResultArray();
}



// Sakit ============================================================
public function getPresensiSakit($tanggal, $id_kelas = null)
{
    $builder = $this->db->table('tb_siswa s')
        ->select('
            s.nis,
            s.nama_siswa,
            s.jenis_kelamin,
            k.kelas,
            p.jam_masuk,
            p.jam_keluar,
            p.jam_sholat,
            kh.kehadiran,
            p.keterangan
        ')
        ->join('tb_kelas k', 'k.id_kelas = s.id_kelas')
        ->join(
            'tb_presensi_siswa p',
            'p.id_siswa = s.id_siswa AND p.tanggal = "'.$tanggal.'"',
            'left'
        )
        ->join('tb_kehadiran kh', 'kh.id_kehadiran = p.id_kehadiran', 'left')
        ->where('kh.kehadiran','sakit'); // <-- hanya yang hadir (atau izin/sakit)

    if (!empty($id_kelas)) {
        $builder->where('s.id_kelas', $id_kelas);
    }

    return $builder->orderBy('s.nama_siswa', 'ASC')->get()->getResultArray();
}


// Izin ============================================================
public function getPresensiIzin($tanggal, $id_kelas = null)
{
    $builder = $this->db->table('tb_siswa s')
        ->select('
            s.nis,
            s.nama_siswa,
            s.jenis_kelamin,
            k.kelas,
            p.jam_masuk,
            p.jam_keluar,
            p.jam_sholat,
            kh.kehadiran,
            p.keterangan
        ')
        ->join('tb_kelas k', 'k.id_kelas = s.id_kelas')
        ->join(
            'tb_presensi_siswa p',
            'p.id_siswa = s.id_siswa AND p.tanggal = "'.$tanggal.'"',
            'left'
        )
        ->join('tb_kehadiran kh', 'kh.id_kehadiran = p.id_kehadiran', 'left')
        ->where('kh.kehadiran','Izin'); // <-- hanya yang hadir (atau izin/sakit)

    if (!empty($id_kelas)) {
        $builder->where('s.id_kelas', $id_kelas);
    }

    return $builder->orderBy('s.nama_siswa', 'ASC')->get()->getResultArray();
}












    /**
 * Ambil data presensi HARIAN - SISWA SAKIT
 */
public function getPresensiHarianSakit($tanggal, $id_kelas = null)
{
    $builder = $this->db->table('tb_siswa s')
        ->select('
            s.nis,
            s.nama_siswa,
            s.jenis_kelamin,
            k.kelas,
            p.jam_masuk,
            p.jam_keluar,
            p.jam_sholat,
            kh.kehadiran,
            p.keterangan
        ')
        ->join('tb_kelas k', 'k.id_kelas = s.id_kelas')
        ->join(
            'tb_presensi_siswa p',
            'p.id_siswa = s.id_siswa AND p.tanggal = "'.$tanggal.'"',
            'left'
        )
        ->join('tb_kehadiran kh', 'kh.id_kehadiran = p.id_kehadiran', 'left')
        ->where('kh.kehadiran', 'Sakit'); // filter sakit

    if (!empty($id_kelas)) {
        $builder->where('s.id_kelas', $id_kelas);
    }

    return $builder
        ->orderBy('s.nama_siswa', 'ASC')
        ->get()
        ->getResultArray();
}





    // // 🔥 ====================================================
    // ==========================================================
    public function getPresensiBulanan($bulan, $id_kelas = null)
    {
        $builder = $this->db->table('tb_siswa s')
            ->select('
                s.nis,
                s.nama_siswa,
                k.kelas,

                SUM(CASE WHEN kh.kehadiran = "Hadir" THEN 1 ELSE 0 END) AS hadir,
                SUM(CASE WHEN kh.kehadiran = "Sakit" THEN 1 ELSE 0 END) AS sakit,
                SUM(CASE WHEN kh.kehadiran = "Izin"  THEN 1 ELSE 0 END) AS izin
            ')
            ->join('tb_kelas k', 'k.id_kelas = s.id_kelas')
            ->join(
                'tb_presensi_siswa p',
                'p.id_siswa = s.id_siswa AND DATE_FORMAT(p.tanggal,"%Y-%m")="'.$bulan.'"',
                'left'
            )
            ->join('tb_kehadiran kh', 'kh.id_kehadiran = p.id_kehadiran', 'left')
            ->groupBy('s.id_siswa');

        if (!empty($id_kelas)) {
            $builder->where('s.id_kelas', $id_kelas);
        }

        $result = $builder->get()->getResultArray();

        // 🔥 Alpha hanya sampai hari berjalan + tanpa libur
        foreach ($result as &$row) {
            $total_hari = $this->countHariEfektif($bulan);
            $row['alpha'] = $total_hari - ($row['hadir'] + $row['sakit'] + $row['izin']);

            if ($row['alpha'] < 0) {
                $row['alpha'] = 0;
            }
        }

        return $result;
    }






    /**
     * 🔥 HITUNG HARI EFEKTIF BULANAN
     * - Senin–Jumat
     * - BUKAN hari libur (tb_hari_libur)
     * - HANYA sampai hari berjalan
     */
    private function countHariEfektif($bulan)
    {
        [$year, $month] = explode('-', $bulan);

        $today = date('Y-m-d');
        $endDate = date('Y-m-d', strtotime("$year-$month-01 last day of this month"));

        // 🔥 jika bulan berjalan → hanya sampai hari ini
        if (date('Y-m', strtotime($today)) === $bulan) {
            $endDate = $today;
        }

        // 🔥 ambil hari libur
        $liburModel = new \App\Models\HariLiburModel();
        $hariLibur  = $liburModel->findColumn('tanggal');

        $start = new \DateTime("$year-$month-01");
        $end   = new \DateTime($endDate);

        $total = 0;
        while ($start <= $end) {

            $tanggal = $start->format('Y-m-d');
            $hari    = $start->format('N'); // 1=Senin ... 7=Minggu

            // ✅ Senin–Jumat & bukan hari libur
            if ($hari < 6 && !in_array($tanggal, $hariLibur)) {
                $total++;
            }

            $start->modify('+1 day');
        }

        return $total;
    }





    // // 🔥 ====================================================
    // ==========================================================
    public function getRekapPerSiswa($id_siswa, $bulan)
    {
        return $this->db->table('tb_presensi_siswa p')
            ->select('p.tanggal, p.jam_masuk, p.jam_keluar, p.jam_sholat, kh.kehadiran, p.keterangan')
            ->join('tb_kehadiran kh', 'kh.id_kehadiran = p.id_kehadiran', 'left')
            ->where('p.id_siswa', $id_siswa)
            ->where('DATE_FORMAT(p.tanggal,"%Y-%m")', $bulan)
            ->orderBy('p.tanggal', 'ASC')
            ->get()->getResultArray();
    }





    // // 🔥 ====================================================
    // ==========================================================
    public function getTotalPerSiswa($id_siswa, $bulan)
    {
        $builder = $this->db->table('presensi');

        $builder->select("
            SUM(CASE WHEN kehadiran = 'Hadir' THEN 1 ELSE 0 END) AS hadir,
            SUM(CASE WHEN kehadiran = 'Sakit' THEN 1 ELSE 0 END) AS sakit,
            SUM(CASE WHEN kehadiran = 'Izin' THEN 1 ELSE 0 END) AS izin,
            SUM(
                CASE 
                    WHEN (kehadiran IS NULL OR kehadiran = '') 
                    THEN 1 ELSE 0 
                END
            ) AS alpha
        ");

        $builder->where('id_siswa', $id_siswa);
        $builder->like('tanggal', $bulan, 'after');
        $builder->where('tanggal <=', date('Y-m-d'));
        $builder->where('DAYOFWEEK(tanggal) NOT IN (1,7)');

        return $builder->get()->getRowArray();
    }


    // // 🔥 ====================================================
    // ==========================================================
    private function countHariEfektifMingguan($awal, $akhir, $hariLibur = [])
    {
        $tgl = new \DateTime($awal);
        $end = new \DateTime($akhir);

        $total = 0;
        while ($tgl <= $end) {

            $tanggal = $tgl->format('Y-m-d');
            $hari    = $tgl->format('N');

            if ($hari < 6 && !in_array($tanggal, $hariLibur)) {
                $total++;
            }

            $tgl->modify('+1 day');
        }

        return $total;
    }

    
    
    
    // // 🔥 ====================================================
    // ========================================================== 
    
    public function getPresensiMingguan($tanggalAwal, $tanggalAkhir, $id_kelas = null)
{
    // 🔥 BATASI JIKA MINGGU BERJALAN
    $today = date('Y-m-d');
    if ($tanggalAkhir > $today) {
        $tanggalAkhir = $today;
    }

        $liburModel = new \App\Models\HariLiburModel();
        $hariLibur  = $liburModel->findColumn('tanggal');

        $builder = $this->db->table('tb_siswa s')
            ->select('
                s.nis,
                s.nama_siswa,
                k.kelas,

                SUM(CASE WHEN kh.kehadiran = "Hadir" THEN 1 ELSE 0 END) AS hadir,
                SUM(CASE WHEN kh.kehadiran = "Sakit" THEN 1 ELSE 0 END) AS sakit,
                SUM(CASE WHEN kh.kehadiran = "Izin"  THEN 1 ELSE 0 END) AS izin
            ')
            ->join('tb_kelas k', 'k.id_kelas = s.id_kelas')
            ->join(
                'tb_presensi_siswa p',
                'p.id_siswa = s.id_siswa 
                 AND p.tanggal BETWEEN "'.$tanggalAwal.'" AND "'.$tanggalAkhir.'"',
                'left'
            )
            ->join('tb_kehadiran kh', 'kh.id_kehadiran = p.id_kehadiran', 'left')
            ->groupBy('s.id_siswa');

        if (!empty($id_kelas)) {
            $builder->where('s.id_kelas', $id_kelas);
        }

        $result = $builder->get()->getResultArray();

        $hariEfektif = $this->countHariEfektifMingguan(
            $tanggalAwal,
            $tanggalAkhir,
            $hariLibur
        );

        foreach ($result as &$row) {
            $row['alpha'] = $hariEfektif - ($row['hadir'] + $row['sakit'] + $row['izin']);
            if ($row['alpha'] < 0) {
                $row['alpha'] = 0;
            }
        }

        return $result;
    }
}
